/*****************************************************************************
*  LPC154x WS2811 LED driver demo LPC15xx board
*
*  1. Use SCT1 timer to transmit WS2811 LED driver frames (24 bit)
*  2. SCT1_OUT0 is linked to P0_1
*****************************************************************************/

#include "board.h"
#include "sct1_ws2811.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
volatile int busy;
volatile int count;

const uint16_t pattern[] =
{
    0x123, 0x456,                                      // 24 bit WS2811 frames split into 2 x 12 bits
    0xFF0, 0x0CC,
    0x555, 0x555,
    0x800, 0x001,
};
/*****************************************************************************
 * Private functions
 ****************************************************************************/
void SCT1_Init(void);
/*****************************************************************************
 * Public functions
 **********************************************************/

void LEDDRIVER_open(void)
{

	Chip_SCT_Init(LPC_SCT1);

	LPC_SCT1->STATE_L         = 12;                        // start state
	LPC_SCT1->OUTPUT         &= ~((1 << DATA_OUT)          // preset data output low
	                              |   (1 << AUX_OUT));     // preset aux output low

	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_0, (WS2811_FB - 1));	// full bit period
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_1, (WS2811_T0H - 1));	// preset data output low
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_2, (WS2811_T1H - 1));	// preset aux output low

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_15, ENABLE_ALL_STATES);	 		// happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_15, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH0 	|	  // related to MATCH0_L
                                                                          SCT_COMBMODE_MATCH	|	  // match only
                                                                          SCT_STATELD_0			  |
                                                                          (31 << 15)			    ));  // add value to STATE



	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_14, 0x00000FFE);	 			// all data bit states except state 0
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_14, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH2 	|	// related to MATCH2_L
                                                                          SCT_COMBMODE_MATCH	));	// match only

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_13, 0x00000FFF);	 			// all data bit states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_13, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1 	|	// related to MATCH1_L
                                                                          SCT_COMBMODE_MATCH	));	// match only

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_12, 0);	 						// contains the 12 bits written by the app
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_12, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1 	|	// related to MATCH1_L
                                                                          SCT_OUTSEL_H			  |	// use OUTPUT for I/O condition
                                                                          (AUX_OUT << 6)		  |	// Use AUX output signal
                                                                          SCT_IOCOND_LOW		  |	// AUX = 0
                                                                          SCT_COMBMODE_AND		));	// match AND I/O

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_11, 0);	 						// contains the 12 bits written by the app
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_11, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1 	|	// related to MATCH1_L
                                                                          SCT_OUTSEL_H			  |	// related to MATCH1_L
                                                                          (AUX_OUT << 6)		  |	// use OUTPUT for I/O condition
                                                                          SCT_IOCOND_HIGH		  |	// AUX = 0
                                                                          SCT_COMBMODE_AND		));	// match AND I/O

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_10,ENABLE_STATE0);	 			// contains the 12 bits written by the app
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_10, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH2 	|	// related to MATCH2_L
                                                                          SCT_COMBMODE_MATCH	|	// match only
                                                                          SCT_STATELD_1			  |	// set STATE to a value
                                                                          SCT_STATEEV_12		  ));	// set STATE to 12




	    LPC_SCT1->OUT[AUX_OUT].SET  = (1 << 10);               // Event 10 toggles the AUX signal
	    LPC_SCT1->OUT[AUX_OUT].CLR  = (1 << 10);               // Event 10 toggles the AUX signal

	    LPC_SCT1->OUT[DATA_OUT].SET = (1 << 15)                // Event 15 sets the DATA signal
	                                   | (1 << 12)             // Event 12 sets the DATA signal
	                                   | (1 << 11);            // Event 11 sets the DATA signal
	    LPC_SCT1->OUT[DATA_OUT].CLR = (1 << 14)                // Event 14 clears the DATA signal
	                                   | (1 << 13)             // Event 13 clears the DATA signal
	                                   | (1 << 10);            // Event 10 clears the DATA signal


	    Chip_SCT_SetConflictResolution(LPC_SCT1, 0, DATA_OUT);	// on conflict: DATA signal doesn't change
	    Chip_SCT_SetConflictResolution(LPC_SCT1, 3, AUX_OUT);	// on conflict: AUX signal toggles

	    LPC_SCT1->LIMIT_L = (1 << 15);                         	// Event 15 limits the counter

	    LPC_SCT1->EVFLAG  = (1 << 10);							// clear event 10 irq flag


	    LPC_SCT1->EVEN   |= (1 << 10);							// Event 10 generates an irq

	    NVIC_EnableIRQ(SCT1_IRQn);                             // enable SCT1 interrupt

}

void LEDDRIVER_write(uint32_t rgb)                         // function to write to a transmit buffer
{
    if (LPC_SCT1->OUTPUT & (1 << AUX_OUT))                 // aux output determines which buffer to write
    {
    	Chip_SCT_EventState(LPC_SCT1 , SCT_EVENT_12 , rgb & 0xFFF);
    }
    else
    {
    	Chip_SCT_EventState(LPC_SCT1 , SCT_EVENT_11 , rgb & 0xFFF);
    }
}

void LEDDRIVER_haltAfterFrame(int on)                      // (de)activate HALT after next frame
{
	 LPC_SCT1->HALT_L = (on << 10);                         // if last, event 10 halts the counter
}

void LEDDRIVER_start(void)                                 // start a frame transmission
{
	LPC_SCT1->COUNT_L = - WS2811_FB * 50;                  // guarantee 50 µs min time between transfers
    LPC_SCT1->STATE_L = 0;                                 // start state
    Chip_SCT_ClearControl(LPC_SCT1,SCT_CTRL_HALT_L);	   // unhalt it by clearing bit 2 of the CTRL register                         // start timer H
}



void SCT1_IRQHandler(void)
{

	LPC_SCT1->EVFLAG = (1u << 10);       				// clear interrupt flag

    if(LPC_SCT1->STATE_L == 12)    						// for debug help
    	Chip_GPIO_SetPinToggle(LPC_GPIO, 0, 7);         // toggle pin P0_7 (LED1)


    if (++count < sizeof(pattern)/sizeof(pattern[0]))  // count nr of frames
    {
        LEDDRIVER_write(pattern[count]);               // write next frame
    }
    else
    {
        LEDDRIVER_haltAfterFrame(1);                   // busy with last frame so halt after this one
    }

    if (LPC_SCT1->CTRL_L & 0x04)                       // if halted, transmission completed
    {
        busy = 0;
    }
}

int main(void)
{
	SystemCoreClockUpdate();
	Board_Init();


	Chip_Clock_EnablePeriphClock( (CHIP_SYSCTL_CLOCK_T) ( SYSCTL_CLOCK_SWM	  |				// enable SWM clock
                                                        SYSCTL_CLOCK_GPIO0 ));				// enable SWM GPIO0


	Chip_SWM_MovablePortPinAssign(SWM_SCT1_OUT0_O , 0, 1);			//SCT01OUT0 = P0.1 = blue LED

	Chip_GPIO_SetPortDIROutput(LPC_GPIO, 0, 7);

	LEDDRIVER_open();

    while (1)                                              					// loop forever
    {
    	if (!busy)
    	{
    	    busy  = 1;                                 // transmission pending
    	    count = 0;                                 // used as frame counter
    	    LEDDRIVER_write(pattern[0]);               // preset first data word
    	    LEDDRIVER_haltAfterFrame(0);
    	    LEDDRIVER_start();                         // start transmission
    	 }
    	__WFI();
    }
}
